<?php

namespace app\controllers;

use app\models\Application;
use app\models\Category;
use app\models\Game;
use app\models\SignupForm;
use Yii;
use yii\filters\AccessControl;
use yii\web\Controller;
use yii\web\Response;
use yii\filters\VerbFilter;
use app\models\LoginForm;
use app\models\ContactForm;

class SiteController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'only' => ['logout'],
                'rules' => [
                    [
                        'actions' => ['logout'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'logout' => ['post'],
                ],
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function actions()
    {
        return [
            'error' => [
                'class' => 'yii\web\ErrorAction',
            ],
            'captcha' => [
                'class' => 'yii\captcha\CaptchaAction',
                'fixedVerifyCode' => YII_ENV_TEST ? 'testme' : null,
            ],
        ];
    }

    /**
     * Displays homepage.
     *
     * @return string
     */
    public function actionIndex()
    {
        return $this->render('index');
    }

    /**
     * Login action.
     *
     * @return Response|string
     */
    public function actionLogin()
    {
        if (!Yii::$app->user->isGuest) {
            return $this->goHome();
        }

        $model = new LoginForm();
        if ($model->load(Yii::$app->request->post()) && $model->login()) {
            return $this->goBack();
        }

        $model->password = '';
        return $this->render('login', [
            'model' => $model,
        ]);
    }

    /**
     * Logout action.
     *
     * @return Response
     */
    public function actionLogout()
    {
        Yii::$app->user->logout();

        return $this->goHome();
    }

    /**
     * Displays contact page.
     *
     * @return Response|string
     */
    public function actionContact()
    {
        $model = new ContactForm();
        if ($model->load(Yii::$app->request->post()) && $model->contact(Yii::$app->params['adminEmail'])) {
            Yii::$app->session->setFlash('contactFormSubmitted');

            return $this->refresh();
        }
        return $this->render('contact', [
            'model' => $model,
        ]);
    }

    /**
     * Displays about page.
     *
     * @return string
     */
    public function actionAbout()
    {
        return $this->render('about');
    }

    public function actionSignup()
    {
        $model = new SignupForm();
        if ($model->load(Yii::$app->request->post()) && $model->signup()) {
            Yii::$app->session->setFlash('success', 'Вы успешно прошли регистрацию!');
            return $this->goHome();
        }
        return $this->render('signup', [
            'model' => $model,
        ]);
    }


    public function actionApplication()
    {
        if (Yii::$app->user->isGuest) {
            return $this->redirect(['site/login']);
        }

        // Получаем все категории
        $categories = Category::find()
            ->select(['name', 'id'])
            ->indexBy('id')
            ->column();

        // Получаем ВСЕ игры сразу
        $allGames = Game::find()
            ->select(['id', 'name', 'category_id'])
            ->asArray()
            ->all();

        // Группируем игры по категориям
        $gamesByCategory = [];
        foreach ($allGames as $game) {
            $gamesByCategory[$game['category_id']][$game['id']] = $game['name'];
        }

        // Создаем новую модель для формы
        $model = new Application();
        $model->user_id = Yii::$app->user->id;

        // Заполняем данные пользователя из профиля
        $user = Yii::$app->user->identity;
        if ($user) {
            if (isset($user->surname)) $model->surname = $user->surname;
            if (isset($user->name)) $model->name = $user->name;
            if (isset($user->patronymic)) $model->patronymic = $user->patronymic;
            if (isset($user->email)) $model->email = $user->email;
            if (isset($user->phone_number)) $model->phone_number = $user->phone_number;
        }

        // Получаем параметры из URL
        $gameId = Yii::$app->request->get('game_id');
        $date = Yii::$app->request->get('date');
        $time = Yii::$app->request->get('time');

        // Автозаполняем поля если пришли параметры
        if ($gameId) {
            $model->game_id = $gameId;
            $game = Game::findOne($gameId);
            if ($game && $game->category_id) {
                $model->category_id = $game->category_id;
            }
        }

        if ($date) {
            $model->date = $date;
        }

        if ($time) {
            $model->time = $time;
            $hour = (int) explode(':', $time)[0];
            if ($hour < 12) {
                $model->price = 1000;
            } elseif ($hour < 16) {
                $model->price = 1500;
            } else {
                $model->price = 2000;
            }
        }

        // Получаем игры для текущей категории (если выбрана)
        $games = [];
        if ($model->category_id && isset($gamesByCategory[$model->category_id])) {
            $games = $gamesByCategory[$model->category_id];
        }

        if ($model->load(Yii::$app->request->post())) {
            $model->user_id = Yii::$app->user->id;

            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Ваша бронь прошла успешно.');

                // После успешного сохранения создаем новую чистую модель
                $model = new Application();
                $model->user_id = Yii::$app->user->id;

                // Заново заполняем данные пользователя
                if ($user) {
                    if (isset($user->surname)) $model->surname = $user->surname;
                    if (isset($user->name)) $model->name = $user->name;
                    if (isset($user->patronymic)) $model->patronymic = $user->patronymic;
                    if (isset($user->email)) $model->email = $user->email;
                    if (isset($user->phone_number)) $model->phone_number = $user->phone_number;
                }

                // Обновляем список игр для новой модели
                $games = [];

                // Используем return $this->refresh() чтобы сбросить все GET параметры
                return $this->refresh();
            }
        }

        return $this->render('application', [
            'model' => $model,
            'categories' => $categories,
            'games' => $games,
            'gamesByCategory' => $gamesByCategory,
        ]);
    }

}