<?php

use yii\helpers\Html;
use yii\widgets\ActiveForm;
use yii\widgets\MaskedInput;
use yii\web\View;

$this->registerCss(<<<CSS
/* Стили для мини-календаря */
.mini-calendar {
    background: #fff;
    border: 1px solid #ddd;
    border-radius: 8px;
    padding: 15px;
    margin-bottom: 20px;
}

.calendar-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 15px;
    padding-bottom: 10px;
    border-bottom: 1px solid #eee;
}

.calendar-month {
    font-weight: bold;
    font-size: 18px;
    color: #333;
}

.calendar-nav-btn {
    background: none;
    border: 1px solid #ddd;
    border-radius: 4px;
    padding: 5px 10px;
    cursor: pointer;
    transition: all 0.3s;
}

.calendar-nav-btn:hover {
    background: #f5f5f5;
}

.calendar-week-days {
    display: grid;
    grid-template-columns: repeat(7, 1fr);
    text-align: center;
    margin-bottom: 10px;
    font-weight: bold;
    color: #666;
}

.calendar-week-days div {
    padding: 5px;
}

.calendar-days {
    display: grid;
    grid-template-columns: repeat(7, 1fr);
    gap: 5px;
}

.calendar-day {
    aspect-ratio: 1;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    cursor: pointer;
    transition: all 0.2s;
    font-size: 14px;
    border: 2px solid transparent;
}

.calendar-day:hover {
    background: #e3f2fd;
    transform: scale(1.1);
}

.calendar-day.empty {
    visibility: hidden;
}

.calendar-day.past {
    color: #ccc;
    cursor: not-allowed;
}

.calendar-day.past:hover {
    background: transparent;
    transform: none;
}

.calendar-day.today {
    border-color: #2196f3;
    background: #e3f2fd;
    font-weight: bold;
}

.calendar-day.selected {
    background: #4CAF50;
    color: white;
}

.calendar-day.weekend {
    color: #dc3545;
}

.time-slots-container {
    margin-top: 20px;
    padding-top: 15px;
    border-top: 1px solid #eee;
}

.time-slots-title {
    font-weight: bold;
    margin-bottom: 10px;
    color: #333;
}

.time-slots {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 8px;
}

.time-slot {
    padding: 8px 5px;
    text-align: center;
    background: #f8f9fa;
    border: 1px solid #dee2e6;
    border-radius: 4px;
    cursor: pointer;
    transition: all 0.2s;
    font-size: 14px;
}

.time-slot:hover {
    background: #e9ecef;
    border-color: #007bff;
}

.time-slot.selected {
    background: #007bff;
    color: white;
    border-color: #007bff;
}

.time-slot.morning {
    border-left: 3px solid #4CAF50;
}

.time-slot.day {
    border-left: 3px solid #2196F3;
}

.time-slot.evening {
    border-left: 3px solid #9C27B0;
}

.price-indicator {
    font-size: 11px;
    color: #666;
    margin-top: 2px;
}

.selected-info {
    background: #e8f5e9;
    padding: 15px;
    border-radius: 8px;
    margin-top: 15px;
    border: 1px solid #4CAF50;
}

.selected-info h5 {
    margin-top: 0;
    color: #2e7d32;
}

.selected-info .btn {
    margin-top: 10px;
}

.form-group {
    margin-bottom: 20px;
}


CSS
);

// Преобразуем PHP массив в JSON для JavaScript
$gamesJson = json_encode($gamesByCategory, JSON_UNESCAPED_UNICODE);

$js = <<<JS
$(document).ready(function() {
    // Данные игр по категориям (из PHP)
    var gamesByCategory = $gamesJson;
    
    // ===================== КАЛЕНДАРЬ =====================
    var currentDate = new Date();
    var currentMonth = currentDate.getMonth();
    var currentYear = currentDate.getFullYear();
    var selectedDate = null;
    var selectedTime = null;
    
    function initCalendar(month, year) {
        var monthNames = [
            'Январь', 'Февраль', 'Март', 'Апрель', 'Май', 'Июнь',
            'Июль', 'Август', 'Сентябрь', 'Октябрь', 'Ноябрь', 'Декабрь'
        ];
        $('.calendar-month').text(monthNames[month] + ' ' + year);
        
        $('.calendar-days').empty();
        
        var firstDay = new Date(year, month, 1);
        var startingDay = firstDay.getDay();
        startingDay = startingDay === 0 ? 6 : startingDay - 1;
        
        var monthLength = new Date(year, month + 1, 0).getDate();
        
        for (var i = 0; i < startingDay; i++) {
            $('.calendar-days').append('<div class="calendar-day empty"></div>');
        }
        
        var today = new Date();
        today.setHours(0, 0, 0, 0);
        
        for (var day = 1; day <= monthLength; day++) {
            var date = new Date(year, month, day);
            var dayClass = 'calendar-day';
            var dayOfWeek = date.getDay();
            
            if (dayOfWeek === 0 || dayOfWeek === 6) {
                dayClass += ' weekend';
            }
            
            if (date.getTime() === today.getTime()) {
                dayClass += ' today';
            }
            
            if (date < today) {
                dayClass += ' past';
            }
            
            if (selectedDate && 
                date.getDate() === selectedDate.getDate() &&
                date.getMonth() === selectedDate.getMonth() &&
                date.getFullYear() === selectedDate.getFullYear()) {
                dayClass += ' selected';
            }
            
            $('.calendar-days').append(
                '<div class="' + dayClass + '" data-date="' + 
                year + '-' + ('0' + (month + 1)).slice(-2) + '-' + 
                ('0' + day).slice(-2) + '">' + day + '</div>'
            );
        }
        
        updateDayClickHandlers();
    }
    
    function updateDayClickHandlers() {
        $('.calendar-day:not(.empty):not(.past)').off('click').on('click', function() {
            var dateStr = $(this).data('date');
            selectedDate = new Date(dateStr);
            selectedDate.setHours(0, 0, 0, 0);
            
            $('.calendar-day').removeClass('selected');
            $(this).addClass('selected');
            
            $('#application-date').val(dateStr);
            
            showTimeSlots(dateStr);
            updateSelectedInfo();
        });
    }
    
    function showTimeSlots(dateStr) {
        var dayOfWeek = new Date(dateStr).getDay();
        var isWeekend = dayOfWeek === 0 || dayOfWeek === 6;
        
        $('.time-slots').empty();
        
        var timeSlots = [];
        if (isWeekend) {
            timeSlots = [
                {time: '09:00', price: 1000, class: 'morning'},
                {time: '11:00', price: 1000, class: 'morning'},
                {time: '13:00', price: 1500, class: 'day'},
                {time: '15:00', price: 1500, class: 'day'},
                {time: '17:00', price: 2000, class: 'evening'},
                {time: '19:00', price: 2000, class: 'evening'},
                {time: '21:00', price: 2500, class: 'evening'}
            ];
        } else {
            timeSlots = [
                {time: '10:00', price: 1000, class: 'morning'},
                {time: '12:00', price: 1000, class: 'morning'},
                {time: '14:00', price: 1500, class: 'day'},
                {time: '16:00', price: 1500, class: 'day'},
                {time: '18:00', price: 2000, class: 'evening'},
                {time: '20:00', price: 2000, class: 'evening'}
            ];
        }
        
        timeSlots.forEach(function(slot) {
            var isSelected = selectedTime === slot.time;
            var slotHtml = '<div class="time-slot ' + slot.class + 
                          (isSelected ? ' selected' : '') + 
                          '" data-time="' + slot.time + 
                          '" data-price="' + slot.price + '">' +
                          slot.time + 
                          '<div class="price-indicator">' + slot.price + ' руб.</div>' +
                          '</div>';
            $('.time-slots').append(slotHtml);
        });
        
        $('.time-slots-container').show();
        
        $('.time-slot').off('click').on('click', function() {
            selectedTime = $(this).data('time');
            var price = $(this).data('price');
            
            $('.time-slot').removeClass('selected');
            $(this).addClass('selected');
            
            $('#application-time').val(selectedTime);
            $('#application-price').val(price);
            
            updateSelectedInfo();
        });
    }
    
    function updateSelectedInfo() {
        if (selectedDate && selectedTime) {
            var dateStr = selectedDate.getFullYear() + '-' + 
                         ('0' + (selectedDate.getMonth() + 1)).slice(-2) + '-' + 
                         ('0' + selectedDate.getDate()).slice(-2);
            var dateFormatted = ('0' + selectedDate.getDate()).slice(-2) + '.' + 
                               ('0' + (selectedDate.getMonth() + 1)).slice(-2) + '.' + 
                               selectedDate.getFullYear();
            
            var price = $('.time-slot.selected').data('price') || '';
            
            var infoHtml = '<h5><i class="fas fa-calendar-check"></i> Выбрано:</h5>' +
                          '<p><strong>Дата:</strong> ' + dateFormatted + '<br>' +
                          '<strong>Время:</strong> ' + selectedTime + '<br>' +
                          '<strong>Цена:</strong> ' + price + ' руб.</p>' +
                          '<button type="button" class="btn btn-sm btn-outline-secondary" id="clear-selection">' +
                          '<i class="fas fa-times"></i> Сбросить выбор</button>';
            
            $('.selected-info').html(infoHtml).show();
            
            $('#clear-selection').off('click').on('click', function() {
                clearSelection();
            });
        } else {
            $('.selected-info').hide();
        }
    }
    
    function clearSelection() {
        selectedDate = null;
        selectedTime = null;
        
        $('.calendar-day').removeClass('selected');
        $('.time-slot').removeClass('selected');
        
        $('#application-date').val('');
        $('#application-time').val('');
        $('#application-price').val('');
        
        $('.selected-info').hide();
        $('.time-slots-container').hide();
    }
    
    $('.prev-month').off('click').on('click', function() {
        currentMonth--;
        if (currentMonth < 0) {
            currentMonth = 11;
            currentYear--;
        }
        initCalendar(currentMonth, currentYear);
    });
    
    $('.next-month').off('click').on('click', function() {
        currentMonth++;
        if (currentMonth > 11) {
            currentMonth = 0;
            currentYear++;
        }
        initCalendar(currentMonth, currentYear);
    });
    
    initCalendar(currentMonth, currentYear);
    $('.time-slots-container').hide();
    $('.selected-info').hide();
    
    // Предзаполнение если есть значения в форме
    var presetDate = $('#application-date').val();
    var presetTime = $('#application-time').val();
    
    if (presetDate) {
        selectedDate = new Date(presetDate);
        selectedDate.setHours(0, 0, 0, 0);
        
        currentMonth = selectedDate.getMonth();
        currentYear = selectedDate.getFullYear();
        initCalendar(currentMonth, currentYear);
        
        showTimeSlots(presetDate);
        
        if (presetTime) {
            selectedTime = presetTime;
            $('.time-slot[data-time="' + presetTime + '"]').addClass('selected');
        }
        
        updateSelectedInfo();
    }
    
    // ===================== ЗАВИСИМЫЙ СПИСОК ИГР =====================
    $('#application-category_id').change(function() {
        var categoryId = $(this).val();
        var gameSelect = $('#application-game_id');
        
        if (!categoryId) {
            gameSelect.html('<option value="">Выберите игру</option>');
            return;
        }
        
        // Получаем игры для выбранной категории
        var games = gamesByCategory[categoryId] || {};
        
        // Сохраняем текущее выбранное значение
        var currentValue = gameSelect.val();
        
        var options = '<option value="">Выберите игру</option>';
        
        $.each(games, function(id, name) {
            var selected = (id == currentValue) ? ' selected' : '';
            options += '<option value="' + id + '"' + selected + '>' + name + '</option>';
        });
        
        gameSelect.html(options);
        
        // Если только одна игра, выбираем ее автоматически
        if (Object.keys(games).length === 1) {
            var gameId = Object.keys(games)[0];
            gameSelect.val(gameId);
        }
    });
    
    // Если категория уже выбрана при загрузке страницы
    var initialCategoryId = $('#application-category_id').val();
    if (initialCategoryId) {
        // Вызываем change событие после небольшой задержки
        setTimeout(function() {
            $('#application-category_id').trigger('change');
        }, 100);
    }
    
    // Если игра уже выбрана, убедимся что она отображается в списке
    var initialGameId = $('#application-game_id').val();
    if (initialGameId && initialCategoryId) {
        var games = gamesByCategory[initialCategoryId] || {};
        if (!games[initialGameId]) {
            // Если выбранной игры нет в текущей категории, очищаем выбор
            $('#application-game_id').val('');
        }
    }
});
JS;

$this->registerJs($js, View::POS_READY);
?>
<div class="d-flex justify-content-center">
    <h3>Бронирование места</h3>
</div>
<?php $form = ActiveForm::begin([
    'id' => 'application-form',
    'options' => ['class' => 'form-horizontal'],
]); ?>

<div class="mini-calendar">
    <div class="calendar-header">
        <button type="button" class="calendar-nav-btn prev-month">
            <i class="fas fa-chevron-left"></i>
        </button>
        <div class="calendar-month"></div>
        <button type="button" class="calendar-nav-btn next-month">
            <i class="fas fa-chevron-right"></i>
        </button>
    </div>

    <div class="calendar-week-days">
        <div>Пн</div>
        <div>Вт</div>
        <div>Ср</div>
        <div>Чт</div>
        <div>Пт</div>
        <div>Сб</div>
        <div>Вс</div>
    </div>

    <div class="calendar-days">
        <!-- Дни будут заполнены через JavaScript -->
    </div>

    <div class="time-slots-container" style="display: none;">
        <div class="time-slots-title">Выберите время:</div>
        <div class="time-slots">
            <!-- Слоты времени будут заполнены через JavaScript -->
        </div>
    </div>

    <div class="selected-info" style="display: none;">
        <!-- Будет заполнено через JavaScript -->
    </div>
</div>

<!-- Поля формы -->
<div class="row">
    <div class="col-md-4">
        <?= $form->field($model, 'surname')->textInput() ?>
    </div>
    <div class="col-md-4">
        <?= $form->field($model, 'name')->textInput() ?>
    </div>
    <div class="col-md-4">
        <?= $form->field($model, 'patronymic')->textInput() ?>
    </div>
</div>

<div class="row">
    <div class="col-md-6">
        <?= $form->field($model, 'phone_number')->widget(MaskedInput::class, [
            'mask' => '8(999)-999-99-99',
        ]) ?>
    </div>
    <div class="col-md-6">
        <?= $form->field($model, 'email')->textInput() ?>
    </div>
</div>

<div class="row">
    <div class="col-md-6">
        <?= $form->field($model, 'category_id')->dropDownList(
            $categories,
            [
                'prompt' => 'Выберите категорию',
                'id' => 'application-category_id'
            ]
        ) ?>
    </div>
    <div class="col-md-6">
        <?= $form->field($model, 'game_id')->dropDownList(
            $games,
            [
                'prompt' => 'Сначала выберите категорию',
                'id' => 'application-game_id'
            ]
        ) ?>
        <?= $form->field($model, 'adres')->textInput() ?>
    </div>
</div>

<div class="row">
    <div class="col-md-4">
        <?= $form->field($model, 'date')->input('date', [
            'readonly' => true,
            'class' => 'form-control bg-light',
            'id' => 'application-date'
        ]) ?>
    </div>
    <div class="col-md-4">
        <?= $form->field($model, 'time')->input('time', [
            'readonly' => true,
            'class' => 'form-control bg-light',
            'id' => 'application-time'
        ]) ?>
    </div>
    <div class="col-md-4">
        <?= $form->field($model, 'price')->textInput([
            'id' => 'application-price',
            'readonly' => true,
            'class' => 'form-control bg-light'
        ]) ?>
    </div>
</div>

<div class="row">
    <div class="col-md-6">
        <?= $form->field($model, 'status_player')->radioList([
            'Новичок' => 'Новичок',
            'Любитель' => 'Любитель',
            'Профессионал' => 'Профессионал',
        ]) ?>
    </div>
    <div class="col-md-6">
        <?= $form->field($model, 'payment_method')->radioList([
            '0' => 'Банковской картой',
            '1' => 'Наличными',
            '2' => 'Переводом',
        ]) ?>
    </div>
</div>
<p>Свои заявки вы сможете посмотреть у себя в профиле.</p>
<div class="form-group">
    <div class="col-lg-offset-1 col-lg-11">
        <?= Html::submitButton('Отправить', ['class' => 'btn btn-primary btn-lg']) ?>
    </div>
</div>

<?php ActiveForm::end(); ?>
<style>
    /* Основные стили страницы */
    body {
        background: linear-gradient(135deg, #f5f7fa 0%, #e4edf5 100%);
        min-height: 100vh;
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    .container {
        max-width: 1200px;
        margin: 0 auto;
        padding: 20px;
    }

    /* Заголовок */
    .d-flex.justify-content-center h3 {
        color: #2c3e50;
        font-weight: 700;
        text-transform: uppercase;
        letter-spacing: 1.5px;
        margin: 30px 0 40px;
        padding: 25px 50px;
        background: linear-gradient(135deg, #ffffff, #f8f9fa);
        border-radius: 25px;
        box-shadow: 0 10px 35px rgba(0, 0, 0, 0.1);
        border-left: 6px solid #3498db;
        border-right: 6px solid #3498db;
        position: relative;
        overflow: hidden;
        text-align: center;
    }

    .d-flex.justify-content-center h3::before {
        content: '🎮 ♟️ 🎲 🃏 🎯';
        position: absolute;
        right: 20px;
        top: 50%;
        transform: translateY(-50%);
        opacity: 0.1;
        font-size: 35px;
        white-space: nowrap;
    }

    /* Форма */
    #application-form {
        background: white;
        border-radius: 30px;
        padding: 50px;
        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
        border: 3px solid #e0e0e0;
        position: relative;
        overflow: hidden;
        margin-bottom: 50px;
    }

    #application-form::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 10px;
        background: linear-gradient(90deg,
        #3498db 0%,
        #9b59b6 25%,
        #e74c3c 50%,
        #f39c12 75%,
        #2ecc71 100%);
        z-index: 1;
    }

    /* Улучшенный мини-календарь в тематике игр */
    .mini-calendar {
        background: linear-gradient(145deg, #ffffff, #f8f9fa);
        border: 3px solid #dfe6e9;
        border-radius: 25px;
        padding: 30px;
        margin-bottom: 40px;
        box-shadow: 0 15px 40px rgba(52, 152, 219, 0.2);
        position: relative;
        z-index: 2;
    }

    .mini-calendar::after {
        content: '📅 🗓️ 📆';
        position: absolute;
        bottom: 20px;
        right: 25px;
        opacity: 0.08;
        font-size: 32px;
        z-index: 0;
    }

    .calendar-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 25px;
        padding-bottom: 20px;
        border-bottom: 3px solid #e0e0e0;
        position: relative;
        z-index: 1;
    }

    .calendar-month {
        font-weight: 800;
        font-size: 26px;
        color: #2c3e50;
        text-transform: uppercase;
        letter-spacing: 2px;
        text-shadow: 1px 1px 3px rgba(0,0,0,0.1);
    }

    .calendar-nav-btn {
        background: linear-gradient(135deg, #3498db, #2980b9);
        border: none;
        border-radius: 50%;
        width: 45px;
        height: 45px;
        color: white;
        cursor: pointer;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        justify-content: center;
        box-shadow: 0 6px 15px rgba(52, 152, 219, 0.4);
        font-size: 16px;
    }

    .calendar-nav-btn:hover {
        transform: scale(1.15) rotate(5deg);
        box-shadow: 0 10px 25px rgba(52, 152, 219, 0.6);
        background: linear-gradient(135deg, #2980b9, #21618c);
    }

    .calendar-week-days {
        display: grid;
        grid-template-columns: repeat(7, 1fr);
        text-align: center;
        margin-bottom: 20px;
        font-weight: 700;
        color: #3498db;
        background: linear-gradient(135deg, #f0f7ff, #e3f2fd);
        border-radius: 15px;
        padding: 15px 5px;
        border: 2px solid #d4e6f1;
    }

    .calendar-week-days div {
        padding: 12px 5px;
        font-size: 15px;
    }

    .calendar-days {
        display: grid;
        grid-template-columns: repeat(7, 1fr);
        gap: 12px;
    }

    .calendar-day {
        aspect-ratio: 1;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
        cursor: pointer;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        font-size: 17px;
        border: 3px solid transparent;
        background: white;
        font-weight: 600;
        box-shadow: 0 3px 10px rgba(0,0,0,0.08);
        position: relative;
        z-index: 1;
    }

    .calendar-day:hover {
        background: linear-gradient(135deg, #3498db, #2980b9);
        color: white;
        transform: translateY(-5px) scale(1.15);
        box-shadow: 0 10px 25px rgba(52, 152, 219, 0.5);
        z-index: 2;
    }

    .calendar-day.empty {
        visibility: hidden;
    }

    .calendar-day.past {
        color: #bdc3c7;
        background: #f8f9fa;
        cursor: not-allowed;
        box-shadow: none;
        opacity: 0.6;
    }

    .calendar-day.past:hover {
        background: #f8f9fa;
        color: #bdc3c7;
        transform: none;
        box-shadow: none;
    }

    .calendar-day.today {
        border-color: #f39c12;
        background: linear-gradient(135deg, #fff9e6, #ffeaa7);
        font-weight: bold;
        color: #e67e22;
        animation: pulse 2s infinite;
    }

    @keyframes pulse {
        0% { box-shadow: 0 0 0 0 rgba(243, 156, 18, 0.4); }
        70% { box-shadow: 0 0 0 10px rgba(243, 156, 18, 0); }
        100% { box-shadow: 0 0 0 0 rgba(243, 156, 18, 0); }
    }

    .calendar-day.weekend {
        color: #e74c3c;
        background: linear-gradient(135deg, #ffeaea, #ffcccc);
        border-color: #ffcccc;
    }

    .calendar-day.selected {
        background: linear-gradient(135deg, #2ecc71, #27ae60);
        color: white;
        font-weight: bold;
        border-color: #27ae60;
        box-shadow: 0 8px 25px rgba(46, 204, 113, 0.5);
        transform: scale(1.15);
        position: relative;
    }

    .calendar-day.selected::after {
        content: '✓';
        position: absolute;
        top: 2px;
        right: 2px;
        font-size: 12px;
        background: white;
        color: #27ae60;
        width: 18px;
        height: 18px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
    }

    /* Слоты времени */
    .time-slots-container {
        margin-top: 30px;
        padding-top: 25px;
        border-top: 3px solid #dfe6e9;
        position: relative;
    }

    .time-slots-title {
        font-weight: 800;
        margin-bottom: 20px;
        color: #2c3e50;
        font-size: 20px;
        display: flex;
        align-items: center;
        gap: 12px;
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    .time-slots-title::before {
        content: '🕐 ⏰ 🕒';
        font-size: 24px;
    }

    .time-slots {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 15px;
    }

    .time-slot {
        padding: 15px 10px;
        text-align: center;
        background: white;
        border: 3px solid #dfe6e9;
        border-radius: 15px;
        cursor: pointer;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        font-size: 16px;
        font-weight: 600;
        position: relative;
        overflow: hidden;
    }

    .time-slot::before {
        content: '';
        position: absolute;
        left: 0;
        top: 0;
        bottom: 0;
        width: 6px;
        background: #3498db;
        z-index: 1;
    }

    .time-slot::after {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: linear-gradient(135deg, transparent, rgba(255,255,255,0.1));
        opacity: 0;
        transition: opacity 0.3s;
    }

    .time-slot:hover {
        background: linear-gradient(135deg, #e3f2fd, #bbdefb);
        border-color: #3498db;
        transform: translateY(-5px);
        box-shadow: 0 10px 25px rgba(52, 152, 219, 0.3);
    }

    .time-slot:hover::after {
        opacity: 1;
    }

    .time-slot.selected {
        background: linear-gradient(135deg, #3498db, #2980b9);
        color: white;
        border-color: #2980b9;
        transform: translateY(-5px) scale(1.05);
        box-shadow: 0 15px 35px rgba(52, 152, 219, 0.5);
    }

    .time-slot.selected::before {
        background: white;
    }

    .time-slot.selected::after {
        background: linear-gradient(135deg, transparent, rgba(255,255,255,0.2));
        opacity: 1;
    }

    .time-slot.morning::before {
        background: linear-gradient(to bottom, #4CAF50, #81C784);
    }

    .time-slot.day::before {
        background: linear-gradient(to bottom, #2196F3, #64B5F6);
    }

    .time-slot.evening::before {
        background: linear-gradient(to bottom, #9C27B0, #BA68C8);
    }

    .price-indicator {
        font-size: 13px;
        color: #7f8c8d;
        margin-top: 8px;
        font-weight: 500;
        position: relative;
        z-index: 2;
    }

    .time-slot.selected .price-indicator {
        color: rgba(255, 255, 255, 0.95);
        font-weight: 600;
    }

    /* Блок выбранной информации */
    .selected-info {
        background: linear-gradient(135deg, #e8f5e9, #c8e6c9);
        padding: 25px;
        border-radius: 20px;
        margin-top: 25px;
        border: 3px solid #4CAF50;
        box-shadow: 0 15px 35px rgba(46, 204, 113, 0.25);
        position: relative;
        overflow: hidden;
    }

    .selected-info::before {
        content: '✅ 🎯 🎉';
        position: absolute;
        bottom: 10px;
        right: 15px;
        opacity: 0.1;
        font-size: 28px;
    }

    .selected-info h5 {
        margin-top: 0;
        color: #2e7d32;
        font-weight: 800;
        display: flex;
        align-items: center;
        gap: 12px;
        font-size: 20px;
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    .selected-info h5::before {
        content: '🎮 ВЫБРАНО';
        font-size: 18px;
    }

    .selected-info p {
        margin: 20px 0;
        color: #34495e;
        font-size: 17px;
        line-height: 1.6;
    }

    .selected-info p strong {
        color: #2c3e50;
        font-weight: 700;
        display: inline-block;
        min-width: 150px;
    }

    .selected-info .btn {
        margin-top: 20px;
        background: linear-gradient(135deg, #f44336, #d32f2f);
        border: none;
        border-radius: 25px;
        padding: 12px 25px;
        color: white;
        font-weight: 600;
        transition: all 0.3s;
        text-transform: uppercase;
        letter-spacing: 1px;
        font-size: 14px;
        box-shadow: 0 5px 15px rgba(244, 67, 54, 0.4);
    }

    .selected-info .btn:hover {
        transform: translateY(-3px);
        box-shadow: 0 10px 25px rgba(244, 67, 54, 0.6);
        background: linear-gradient(135deg, #d32f2f, #b71c1c);
    }

    .selected-info .btn::before {
        content: '🗑️ ';
        margin-right: 8px;
    }

    /* Поля формы */
    .form-group {
        margin-bottom: 30px;
        position: relative;
    }

    .form-group label {
        font-weight: 700;
        color: #2c3e50;
        margin-bottom: 10px;
        display: flex;
        align-items: center;
        gap: 12px;
        font-size: 16px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .form-control {
        border: 3px solid #dfe6e9;
        border-radius: 15px;
        padding: 15px 20px;
        font-size: 17px;
        transition: all 0.3s;
        background: #f8f9fa;
        height: auto;
    }

    .form-control:focus {
        border-color: #3498db;
        box-shadow: 0 0 0 0.3rem rgba(52, 152, 219, 0.25);
        background: white;
        transform: translateY(-2px);
    }

    .form-control.bg-light {
        background-color: #f8f9fa !important;
        cursor: default;
        color: #6c757d;
    }

    /* Стили для select */
    select.form-control {
        appearance: none;
        background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='20' height='20' fill='%233498db' viewBox='0 0 16 16'%3E%3Cpath d='M7.247 11.14L2.451 5.658C1.885 5.013 2.345 4 3.204 4h9.592a1 1 0 0 1 .753 1.659l-4.796 5.48a1 1 0 0 1-1.506 0z'/%3E%3C/svg%3E");
        background-repeat: no-repeat;
        background-position: right 20px center;
        background-size: 20px;
        padding-right: 50px;
    }

    /* Радио кнопки */
    .radio-list {
        display: flex;
        flex-wrap: wrap;
        gap: 25px;
        margin-top: 15px;
    }

    .radio-list label {
        display: flex;
        align-items: center;
        padding: 15px 25px;
        background: linear-gradient(135deg, #f8f9fa, #e9ecef);
        border-radius: 15px;
        border: 3px solid #dfe6e9;
        cursor: pointer;
        transition: all 0.3s;
        font-weight: 600;
        color: #495057;
        min-width: 180px;
        justify-content: center;
    }

    .radio-list label:hover {
        border-color: #3498db;
        background: linear-gradient(135deg, #e3f2fd, #bbdefb);
        transform: translateY(-3px);
        box-shadow: 0 8px 20px rgba(52, 152, 219, 0.2);
    }

    .radio-list input[type="radio"]:checked + label {
        background: linear-gradient(135deg, #3498db, #2980b9);
        color: white;
        border-color: #2980b9;
        box-shadow: 0 10px 25px rgba(52, 152, 219, 0.4);
        transform: translateY(-3px);
    }

    .radio-list input[type="radio"] {
        display: none;
    }

    /* Особые стили для радиокнопок статуса игрока */
    .field-application-status_player .radio-list label::before {
        margin-right: 10px;
        font-size: 20px;
    }

    .radio-list label[for="application-status_player-0"]::before {
        content: '👶';
    }

    .radio-list label[for="application-status_player-1"]::before {
        content: '👍';
    }

    .radio-list label[for="application-status_player-2"]::before {
        content: '🏆';
    }

    /* Особые стили для радиокнопок способа оплаты */
    .field-application-payment_method .radio-list label::before {
        margin-right: 10px;
        font-size: 20px;
    }

    .radio-list label[for="application-payment_method-0"]::before {
        content: '💳';
    }

    .radio-list label[for="application-payment_method-1"]::before {
        content: '💰';
    }

    .radio-list label[for="application-payment_method-2"]::before {
        content: '🔄';
    }

    /* Иконки для полей ввода */
    .form-group label::before {
        font-family: "Font Awesome 5 Free";
        font-weight: 900;
        font-size: 18px;
        color: #3498db;
        min-width: 25px;
        text-align: center;
    }

    .form-group.field-application-surname label::before {
        content: "👤";
    }

    .form-group.field-application-name label::before {
        content: "📝";
    }

    .form-group.field-application-patronymic label::before {
        content: "👨‍👩‍👧‍👦";
    }

    .form-group.field-application-phone_number label::before {
        content: "📱";
    }

    .form-group.field-application-email label::before {
        content: "📧";
    }

    .form-group.field-application-category_id label::before {
        content: "📂";
    }

    .form-group.field-application-game_id label::before {
        content: "🎲";
    }

    .form-group.field-application-adres label::before {
        content: "📍";
    }

    .form-group.field-application-date label::before {
        content: "📅";
    }

    .form-group.field-application-time label::before {
        content: "⏰";
    }

    .form-group.field-application-price label::before {
        content: "💰";
    }

    /* Кнопка отправки */
    .btn-primary {
        background: linear-gradient(135deg, #3498db, #2980b9);
        border: none;
        border-radius: 30px;
        padding: 20px 60px;
        font-size: 20px;
        font-weight: 800;
        text-transform: uppercase;
        letter-spacing: 2px;
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        box-shadow: 0 15px 40px rgba(52, 152, 219, 0.5);
        display: block;
        margin: 50px auto 30px;
        position: relative;
        overflow: hidden;
        min-width: 300px;
    }

    .btn-primary::before {
        content: '🎮 Отправить заявку';
        display: block;
    }

    .btn-primary::after {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.4), transparent);
        transition: left 0.8s;
    }

    .btn-primary:hover {
        transform: translateY(-8px) scale(1.05);
        box-shadow: 0 25px 60px rgba(52, 152, 219, 0.7);
        background: linear-gradient(135deg, #2980b9, #21618c);
        color: white;
        text-decoration: none;
    }

    .btn-primary:hover::after {
        left: 100%;
    }

    .btn-primary:active {
        transform: translateY(-4px) scale(1.02);
    }

    /* Адаптивность */
    @media (max-width: 992px) {
        .container {
            padding: 15px;
        }

        #application-form {
            padding: 30px;
        }

        .d-flex.justify-content-center h3 {
            font-size: 22px;
            padding: 20px 30px;
        }

        .calendar-days {
            gap: 8px;
        }

        .calendar-day {
            font-size: 15px;
        }

        .radio-list label {
            min-width: 140px;
        }
    }

    @media (max-width: 768px) {
        #application-form {
            padding: 20px;
        }

        .mini-calendar {
            padding: 20px;
        }

        .calendar-days {
            gap: 5px;
        }

        .calendar-day {
            font-size: 14px;
        }

        .time-slots {
            grid-template-columns: repeat(2, 1fr);
        }

        .radio-list {
            flex-direction: column;
            gap: 15px;
        }

        .radio-list label {
            min-width: 100%;
            justify-content: flex-start;
        }

        .btn-primary {
            padding: 18px 30px;
            font-size: 18px;
            min-width: 100%;
            margin: 40px auto 20px;
        }

        .calendar-month {
            font-size: 20px;
        }
    }

    @media (max-width: 480px) {
        .d-flex.justify-content-center h3 {
            font-size: 18px;
            padding: 15px 20px;
        }

        .calendar-month {
            font-size: 18px;
        }

        .calendar-nav-btn {
            width: 40px;
            height: 40px;
        }

        .time-slots {
            grid-template-columns: 1fr;
        }
    }

    /* Анимации */
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(30px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    @keyframes slideInRight {
        from {
            opacity: 0;
            transform: translateX(30px);
        }
        to {
            opacity: 1;
            transform: translateX(0);
        }
    }

    @keyframes slideInLeft {
        from {
            opacity: 0;
            transform: translateX(-30px);
        }
        to {
            opacity: 1;
            transform: translateX(0);
        }
    }

    #application-form {
        animation: fadeInUp 0.8s ease;
    }

    .mini-calendar {
        animation: slideInRight 1s ease;
    }

    .form-group:nth-child(odd) {
        animation: slideInLeft 1.2s ease;
    }

    .form-group:nth-child(even) {
        animation: slideInRight 1.2s ease;
    }

    /* Декоративные игровые элементы */
    #application-form::after {
        content: '♟️ 🎲 🎮 🃏 🎯';
        position: absolute;
        bottom: 30px;
        right: 30px;
        opacity: 0.05;
        font-size: 60px;
        z-index: 0;
        white-space: nowrap;
    }

    /* Стили для строк формы */
    .row {
        position: relative;
        z-index: 2;
        animation: fadeInUp 0.6s ease;
    }

    .row:nth-child(1) { animation-delay: 0.1s; }
    .row:nth-child(2) { animation-delay: 0.2s; }
    .row:nth-child(3) { animation-delay: 0.3s; }
    .row:nth-child(4) { animation-delay: 0.4s; }
    .row:nth-child(5) { animation-delay: 0.5s; }

    /* Стили для обязательных полей */
    .form-group.required label::after {
        content: ' *';
        color: #e74c3c;
        font-size: 20px;
        margin-left: 5px;
    }

    /* Подсказки */
    .help-block {
        color: #7f8c8d;
        font-size: 14px;
        margin-top: 8px;
        font-style: italic;
    }

    /* Уведомление о валидации */
    .has-error .form-control {
        border-color: #e74c3c;
        background: #ffeaea;
    }

    .has-error .help-block {
        color: #e74c3c;
        font-weight: 500;
    }

    /* Лоадер для загрузки игр */
    #game_id_loader {
        display: none;
        text-align: center;
        padding: 10px;
        color: #3498db;
        font-style: italic;
    }

    /* Информационный блок */
    .alert-info {
        background: linear-gradient(135deg, #e3f2fd, #bbdefb);
        border: 3px solid #3498db;
        border-radius: 20px;
        padding: 25px;
        margin-top: 30px;
    }

    .alert-info h5 {
        color: #2c3e50;
        font-weight: 800;
        display: flex;
        align-items: center;
        gap: 10px;
        font-size: 20px;
    }

    .alert-info h5::before {
        content: 'ℹ️';
    }

    .alert-info ul {
        padding-left: 20px;
        margin-bottom: 0;
    }

    .alert-info li {
        margin-bottom: 10px;
        color: #34495e;
        font-size: 16px;
        position: relative;
        padding-left: 10px;
    }

    .alert-info li:before {
        content: '🎲';
        margin-right: 10px;
    }

    .alert-info hr {
        border-color: #3498db;
        opacity: 0.3;
        margin: 20px 0;
    }
</style>